<?php

namespace Go2B\Tasks\Sync;

use Phalcon\DI as Di;

class CsvZanotti
{
  const IMPORT_FOLDER = '../public/io/import';
  const IMAGES_FOLDER = '../public/img';
  const MAX_LINES = 20000;

  // titlin and crart are not sync'd
  const CSV_NAMES = array(
    'anaage', 'anaart', 'anagra', 'ananaz', 'anatgl', 'anavet', 'anazon', 'artcla', 'budget', 'ctarti',
    'cttest', 'dispon', 'desmer', 'gonota', 'gonotv', 'linmod', 'lstest', 'lsvstgl', 'postgl', 'sequsa',
    'sparti', 'svcorp', 'tabstg', 'tipolo', 'tpgene', 'tpmode', 'tppaga', 'tpport'
  );

  const CLASSES_TO_FIX = array(
    'Anagra', 'Artcla', 'Budget', 'Dispon', 'Desmer', 'Gonota', 'Gonotv', 'Lsvstgl', 'Sequsa', 'Sparti',
    'Svcorp', 'Tabstg', 'Tipolo', 'Titlin', 'Tpgene'
  );

  public static function sync($echo) {
    self::preImport($echo);

    foreach (self::CSV_NAMES as $table) {
      // Get sync file
      $file = self::getSyncFile($table);

      $shouldMove = $table != 'anaart';

      // Check if temp folder exists
      $targetPath = self::IMPORT_FOLDER . '/temp';
      if (!is_dir($targetPath) && !mkdir($targetPath, 0777, true)){
        die("Error creating folder $targetPath");
      }

      // Set path temporary file
      $fixedCsv = $targetPath . '/' . self::mapTables($table) . '.csv';

      //$class = self::mapTables($table);
      //$class = ucfirst($class);

      // If file exists and it is readable
      if (file_exists($file) && ($handle = fopen($file, 'r')) !== FALSE) {
        // BOM as a string for comparison.
        $bom = "\xef\xbb\xbf";

        // Progress file pointer and get first 3 characters to compare to the BOM string.
        if (fgets($handle, 4) !== $bom) {
          // BOM not found - rewind pointer to start of file.
          rewind($handle);
        }

        // Skip first row
        $data = fgetcsv($handle);

        // Get labels
        $labels = array();
        foreach ($data as $label) {
          $labels[] = $label;
        }

        $fp = file($file);
        $totLines = count($fp) - 1;

        if ($totLines < self::MAX_LINES || in_array($table, array('gonota', 'gonotv'))) {
          switch ($table) {
            case 'artcla':
              $classes = array('Artcla', 'Artcrt');
              self::emptyTags();
              self::emptyTableIfNeeded('Artcla');
              self::writeHeaderMultipleCsvs($fixedCsv, $classes);
              self::manageArtcla($totLines, $echo, 1, $handle);
              self::loadFixedMultipleCsvsToDb($fixedCsv, $classes);
              break;
            case 'svcorp':
              $classes = array('Svcorp', 'Svtest', 'Svtipo', 'Svarti', 'Svanag');
              self::emptyStatistics();
              self::writeHeaderMultipleCsvs($fixedCsv, $classes);
              self::createFixedSpecialCasesCsv($fixedCsv, 'svcorp', $classes, $labels, $totLines, $echo, 1, $handle);
              self::loadFixedMultipleCsvsToDb($fixedCsv, $classes);
              break;
            case 'linmod':
              $classes = array('Linmod', 'Titlin', 'Sermod');
              self::writeHeaderMultipleCsvs($fixedCsv, $classes);
              self::createFixedSpecialCasesCsv($fixedCsv, 'linmod', $classes, $labels, $totLines, $echo, 1, $handle);
              self::loadFixedMultipleCsvsToDb($fixedCsv, $classes);
              break;
            case 'gonota':
            case 'gonotv':
              self::emptyTableIfNeeded('Desart');
              self::writeHeaderCsv($fixedCsv, 'Desart');
              self::createFixedCsvForDesart($fixedCsv, 'desart', $totLines, $echo, $table == 'gonota', $handle);
              self::loadFixedCsvToDb($fixedCsv, 'desart');
              break;
            case 'lsvstgl':
              self::emptyTableIfNeeded('Lscorp');
              self::writeHeaderCsv($fixedCsv, 'Lscorp');
              self::createFixedCsv($fixedCsv, 'lscorp', $labels, $totLines, $echo, 1, $handle);
              self::loadFixedCsvToDb($fixedCsv, 'lscorp');
              break;
            default:
              self::emptyTableIfNeeded(self::getClass($table));
              self::writeHeaderCsv($fixedCsv, self::getClass($table));
              $table = self::createFixedCsv($fixedCsv, $table, $labels, $totLines, $echo, 1, $handle);
              self::addCustomRowToCsv($fixedCsv, self::getClass($table));
              self::loadFixedCsvToDb($fixedCsv, $table);
              break;
          }
        } else {
          $numSteps = ceil($totLines / self::MAX_LINES);
          switch ($table) {
            case 'artcla':
              $classes = array('Artcla', 'Artcrt');
              self::emptyTags();
              self::emptyTableIfNeeded('Artcla');
              for ($i = 0; $i < $numSteps; $i++) {
                self::writeHeaderMultipleCsvs($fixedCsv, $classes);
                self::manageArtcla($totLines, $echo, ($i * self::MAX_LINES) + 1, $handle);
                self::loadFixedMultipleCsvsToDb($fixedCsv, $classes);
              }
              break;
            case 'svcorp':
              $classes = array('Svcorp', 'Svtest', 'Svtipo', 'Svarti', 'Svanag');
              self::emptyStatistics();
              for ($i = 0; $i < $numSteps; $i++) {
                self::writeHeaderMultipleCsvs($fixedCsv, $classes);
                self::createFixedSpecialCasesCsv($fixedCsv, 'svcorp', $classes, $labels, $totLines, $echo, ($i * self::MAX_LINES) + 1, $handle);
                self::loadFixedMultipleCsvsToDb($fixedCsv, $classes);
              }
              break;
            case 'linmod':
              $classes = array('Linmod', 'Titlin', 'Sermod');
              for ($i = 0; $i < $numSteps; $i++) {
                self::writeHeaderMultipleCsvs($fixedCsv, $classes);
                self::createFixedSpecialCasesCsv($fixedCsv, 'linmod', $classes, $labels, $totLines, $echo, ($i * self::MAX_LINES) + 1, $handle);
                self::loadFixedMultipleCsvsToDb($fixedCsv, $classes);
              }
              break;
            case 'lsvstgl':
              self::emptyTableIfNeeded('Lscorp');
              for ($i = 0; $i < $numSteps; $i++) {
                self::writeHeaderCsv($fixedCsv, 'Lscorp');
                $table = self::createFixedCsv($fixedCsv, 'lscorp', $labels, $totLines, $echo, ($i * self::MAX_LINES) + 1, $handle);
                self::addCustomRowToCsv($fixedCsv, 'Lscorp');
                self::loadFixedCsvToDb($fixedCsv, 'lscorp');
              }
              break;
            default:
              self::emptyTableIfNeeded(self::getClass($table));
              for ($i = 0; $i < $numSteps; $i++) {
                self::writeHeaderCsv($fixedCsv, self::getClass($table));
                $table = self::createFixedCsv($fixedCsv, $table, $labels, $totLines, $echo, ($i * self::MAX_LINES) + 1, $handle);
                self::addCustomRowToCsv($fixedCsv, self::getClass($table));
                self::loadFixedCsvToDb($fixedCsv, $table);
              }
              break;
          }
        }

        fclose($handle);
        if ($shouldMove) {
          rename($file, str_replace('io/import/', 'io/import/processed/', $file));
        }
      }
    }

    self::postImport();
  }

  private static function emptyStatistics() {
    $array = array('Svanag', 'Svarti', 'Svcorp', 'Svtest', 'Svtipo');
    $class = '\Go2B\Models\\' . ucfirst($class);
    if (in_array($class, $array)) {
      $class::emptyTable();
    }
  }

  private static function emptyTags() {
    \Go2B\Models\Artcla::emptyTags();
    \Go2B\Models\Artcrt::emptyTable();
    \Go2B\Models\Tpcrar::emptyTable();
  }

  private static function emptyTableIfNeeded($class) {
    $array = array(
      'Anazon', 'Artcol', 'Artvar', 'Ctanag', 'Ctetic', 'Desart', 'Imgart', 'Lkcorp', 'Lkimag', 'Lkmode', 'Lktest',
      'Pvcatc', 'Pvcorp', 'Pvgrup', 'Pvnote', 'Pvtest', 'Regqta', 'Regqtm', 'Scacon', 'Tpmode'
    );
    // 'Sparti'
    $class = '\Go2B\Models\\' . ucfirst($class);
    if (in_array($class, $array)) {
      $class::emptyTable();
    } else if ($class == 'Desmer') {
      $class::blockAllShippings();
    } else if ($class == 'Artcla') {
      $class::deleteAllAvailabilities();
    }
  }

  private static function writeHeaderCsv($fixedCsv, $class) {
    if ($class != 'B2bBudget') {
      $header = '';
      $class = '\Go2B\Models\\' . ucfirst($class);
      foreach (get_object_vars(new $class) as $key => $value) {
        $header .= $key . ',';
      }
    } else {
      $header = 'tpanag,cdanag,cdtitl,cdlinm,cdserm,cduscc,cdstg0,qtabdg,valbdg,qtatrg,valtrg,cdstg1,qtast1,valst1,cdstg2,qtast2,valst2,cdstg3,qtast3,valst3,cdstg4,qtast4,valst4,';
    }
    file_put_contents($fixedCsv, substr($header, 0, -1) . "\n");
    chmod($fixedCsv, 0777);
  }

  private static function writeHeaderMultipleCsvs($fixedCsv, $classes) {
    foreach ($classes as $class) {
      $fixedCsv = substr($fixedCsv, 0, -10) . strtolower($class) . '.csv';
      self::writeHeaderCsv($fixedCsv, $class);
    }
  }

  private static function writeRowCsv($fixedCsv, $class, $item) {
    if ($class != 'B2bBudget') {
      $row = '';
      $class = '\Go2B\Models\\' . ucfirst($class);
      foreach (get_object_vars(new $class) as $key => $value) {
        $row .= '"' . (isset($item[$key]) ? trim($item[$key]) : '') . '",';
      }
    } else {
      $row = '"' . $item['tpanag'] . '",';
      $row .= '"' . $item['cdanag'] . '",';
      $row .= '"' . $item['cdtitl'] . '",';
      $row .= '"' . $item['cdlinm'] . '",';
      $row .= '"' . $item['cdserm'] . '",';
      $row .= '"' . $item['cduscc'] . '",';
      $row .= '"' . $item['cdstg0'] . '",';
      $row .= '"' . $item['qtabdg'] . '",';
      $row .= '"' . $item['valbdg'] . '",';
      $row .= '"' . $item['qtatrg'] . '",';
      $row .= '"' . $item['valtrg'] . '",';
      $row .= '"' . $item['cdstg1'] . '",';
      $row .= '"' . $item['qtast1'] . '",';
      $row .= '"' . $item['valst1'] . '",';
      $row .= '"' . $item['cdstg2'] . '",';
      $row .= '"' . $item['qtast2'] . '",';
      $row .= '"' . $item['valst2'] . '",';
      $row .= '"' . $item['cdstg3'] . '",';
      $row .= '"' . $item['qtast3'] . '",';
      $row .= '"' . $item['valst3'] . '",';
      $row .= '"",';
      $row .= '"0",';
      $row .= '"0",';
    }
    file_put_contents($fixedCsv, file_get_contents($fixedCsv) . substr($row, 0, -1) . "\n");
  }

  private static function createFixedCsv($fixedCsv, $class, $labels, $totLines, $echo, $offset, &$handle) {
    $i = $offset;

    $initialClass = $class;
    while ($i < (self::MAX_LINES + $offset) && ($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
      $class = $initialClass;

      // Avoid empty rows
      if ($data[0] != '') {
        $item = array();
        for ($j = 0; $j < count($data); $j++) {
          $item[$labels[$j]] = $data[$j];
        }

        if (in_array(ucfirst($class), self::CLASSES_TO_FIX)) {
          self::fixItem($class, $item);
        }

        // Add row to csv
        self::writeRowCsv($fixedCsv, self::getClass($class), $item);

        if ($echo) {
          echo strtolower($class) . ' Row ' . $i . '/' . $totLines . "\n";
          $i++;
        }
      }
    }
    return $class;
  }

  private static function createFixedSpecialCasesCsv($fixedCsv, $file, $classes, $labels, $totLines, $echo, $offset, &$handle) {
    $file = strtolower($file);
    $i = $offset;

    if ($file == 'svcorp') {
      $nextIdCorp = \Go2B\Models\Svcorp::getNextIdcorp();
    }

    while ($i < (self::MAX_LINES + $offset) && ($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
      // Avoid empty rows
      if ($data[0] != '') {
        $item = array();
        for ($j = 0; $j < count($data); $j++) {
          $item[$labels[$j]] = $data[$j];
        }

        self::fixItem($file, $item);

        if ($file == 'svcorp') {
          $item['idcorp'] = $nextIdCorp;
          $nextIdCorp++;
        }

        // Add row to csv
        foreach ($classes as $class) {
          if (strtolower($class) != 'sermod' || $item['cdserm'] != '') {
            $fixedCsv = substr($fixedCsv, 0, -10) . strtolower($class) . '.csv';
            self::writeRowCsv($fixedCsv, $class, $item);
          }
        }

        if ($echo) {
          echo $file . ' Row ' . $i . '/' . $totLines . "\n";
          $i++;
        }
      }
    }
  }

  private static function manageArtcla($totLines, $echo, $offset, &$handle) {
    $i = $offset;
    while ($i < (self::MAX_LINES + $offset) && ($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
      // Avoid empty rows
      if ($data[0] != '') {
        for ($j = 1; $j <= 2; $j++) {
          if ($data[$j] != '') {
            if ($artcla = \Go2B\Models\Artcla::findFirst(array('tpinpu = "AR" AND codice = :codice: AND tpclas = "TB2B"', 'bind' => array('codice' => $data[0])))) {
              $artcla = new \Go2B\Models\Artcla;
              $artcla->tpinpu = 'AR';
              $artcla->codice = $data[0];
              $artcla->tpclas = 'TB2C';
            } else {
              $artcla = new \Go2B\Models\Artcla;
              $artcla->tpinpu = 'AR';
              $artcla->codice = $data[0];
              $artcla->tpclas = 'TB2B';
            }
            $artcla->valore = $data[$j];
            $artcla->save();
          }
        }
        for ($j = 3; $j <= 7; $j++) {
          if ($data[$j] != '') {
            $artcrt = new \Go2B\Models\Artcrt;
            $artcrt->tpinpu = 'AR';
            $artcrt->codice = $data[0];
            $artcrt->tpcrar = 'RAGG';
            $artcrt->valore = $data[$j];
            $artcrt->save();
            $tpcrar = \Go2B\Models\Tpcrar::findFirst(array('tpinpu = "AR" AND tpcrar = "RAGG"'));
            $tpcrar = !empty($tpcrar) ? $tpcrar : new \Go2B\Models\Tpcrar;
            $tpcrar->tpinpu = 'AR';
            $tpcrar->tpcrar = 'RAGG';
            $tpcrar->dscrar = 'Raggruppamento';
            $tpcrar->save();
          }
        }
      }
    }
  }

  private static function createFixedCsvForDesart($fixedCsv, $class, $totLines, $echo, $isGonota, &$handle) {
    $i = 1;

    $initialClass = $class;
    $code = $type = $lang = $value = '';
    while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
      $currCode = $data[0];
      $currType = $data[1];
      $currLang = $data[2];

      if ($currCode != $code || $currType != $type || $currLang != $lang) {
        if ($i != 1 && $currCode != '' && $currType != '' && $currLang != '') {
          // Write to file
          $item = array(
            'tpinpu' => !$isGonota ? 'AR' : 'AN',
            'codice' => $code,
            'tpdesc' => $type,
            'idlang' => $lang,
            'descri' => $value
          );
          self::writeRowCsv($fixedCsv, 'Desart', $item);
        }

        $code = $currCode;
        $type = $currType;
        $lang = $currLang;
        $value = str_replace('"', '\"', $data[4]);
      } else {
        $value .= '<br/>' . str_replace('"', '\"', $data[4]);
      }

      if ($echo) {
        echo strtolower($class) . ' Row ' . $i . '/' . $totLines . "\n";
        $i++;
      }
    }
    return $class;
  }

  private static function addCustomRowToCsv($fixedCsv, $class) {
    $customInsertTables = array('anatgl', 'postgl');
    $table = strtolower($class);
    if (in_array($table, $customInsertTables)) {
      switch ($table) {
        case 'postgl':
          $item = array('cdtagl' => 'UN', 'postgl' => 1, 'taglia' => 'UNI');
          break;
        case 'anatgl':
          $item = array('cdtagl' => 'UN', 'dstagl' => 'UNICA', 'seqrap' => 0);
          break;
      }
      self::writeRowCsv($fixedCsv, $class, $item);
    }
  }

  private static function loadFixedCsvToDb($fixedCsv, $table) {
    $path   = str_replace('\\', '/', realpath($fixedCsv));
    $table  = self::mapTables(strtolower($table));

    $db = Di::getDefault()->get('db');
    if (strtolower($table) != 'anaart') {
      $type   = in_array(strtolower($table),
        ['anagra', 'ananaz', 'anazon', 'artcla', 'artcrt', 'cttest', 'b2b_addinf', 'desart', 'desmer', 'lscorp', 'lstest', 'sparti', 'tipolo', 'tpmode']) ? 'REPLACE' : 'IGNORE';

      if (strtolower($table) == 'artcla' && strpos($fixedCsv, 'dispon.csv') !== false) {
        $db->query('DELETE FROM ' . $table . ' WHERE tpclas = \'DISP\'');
      }

      $query =
        'LOAD DATA LOCAL INFILE "' . $path . '" ' .
        $type . ' INTO TABLE ' . $table . '
        FIELDS TERMINATED BY ","
        ENCLOSED BY \'"\'
        LINES TERMINATED BY "\n"
        IGNORE 1 ROWS';

      $db->query($query);
    } else if (strtolower($table) == 'anaart') {

      //Fix Bug Zanotti Marco 5.9.2022
      //$query = 'UPDATE anaart SET flbloc = 1 WHERE tpvend != "CUSTOM"';
      //$db->query($query);

      $query = 'DROP TABLE IF EXISTS anaart_tmp';
      $db->query($query);
      $query = 'CREATE TABLE anaart_tmp LIKE anaart';
      $db->query($query);
      $query =
        'LOAD DATA LOCAL INFILE "' . $path . '"
        INTO TABLE anaart_tmp
        FIELDS TERMINATED BY ","
        ENCLOSED BY \'"\'
        LINES TERMINATED BY "\n"
        IGNORE 1 ROWS';
      $db->query($query);
      $query =
        'INSERT INTO anaart
        SELECT cdarti, dsarti, cdartn, cdpers, flbloc, cdcolo, dscolo, tpvend, cdmisu, flimag
        FROM anaart_tmp
        ON DUPLICATE KEY UPDATE dsarti = VALUES(dsarti), cdartn = VALUES(cdartn),
          flbloc = VALUES(flbloc), cdcolo = VALUES(cdcolo), dscolo = VALUES(dscolo),
          tpvend = VALUES(tpvend), cdmisu = VALUES(cdmisu)';
      $db->query($query);
      $query = 'DROP TABLE anaart_tmp';
      $db->query($query);
    }

    // Remove temp file
    unlink($fixedCsv);
  }

  private static function loadFixedMultipleCsvsToDb($fixedCsv, $tables) {
    foreach ($tables as $table) {
      $fixedCsv = substr($fixedCsv, 0, -10) . strtolower($table) . '.csv';
      self::loadFixedCsvToDb($fixedCsv, strtolower($table));
    }
  }

  private static function getSyncFile($table) {
    $files = glob(self::IMPORT_FOLDER . '/*' . $table . '*.csv');

    if (count($files) > 0) {
      return $files[0];
    }

    return '';
  }

  private static function getClass($table) {
    /*
    if (substr($table, 0, 4) == 'b2b_') {
      return 'B2b' . ucfirst(substr($table, 4, 6));
    }*/
    if ($table == 'budget' || $table == 'b2b_budget') {
      return 'B2bBudget';
    } else if ($table == 'sequsa' || $table == 'b2b_addinf') {
      return 'B2bAddinf';
    } else if ($table == 'dispon') {
      return 'Artcla';
    }

    return ucfirst($table);
  }

  private static function convertCdlingToIsoIdlang($cdling) {
    switch ($cdling) {
      case 'i':
      case 'I':
      case 'it':
      case 'IT':
      case '1':
        return 'IT';
      case 'e':
      case 'E':
      case 'en':
      case 'EN':
      case '2':
      case '5': // 5 is for russian, but is not supported
        return 'EN';
      case '3':
        return 'FR';
      case '4':
        return 'DE';
      case 'S':
        return 'ES';
    }
  }
  private static function fixItem(&$table, &$item) {
    $table = strtolower($table);
    switch ($table) {
      case 'anagra':
        $item['cdanag'] = $item['sganag'];
        $item['descri'] = $item['descra'] . $item['descrb'];
        $item['cdnazi'] = $item['nazion'];
        $item['idlang'] = self::convertCdlingToIsoIdlang($item['cdling']);
        break;
      case 'budget':
        $item['tpanag'] = 'CL';
        $item['cdserm'] = self::getSeriesCode($item['cdlinm']);
        $item['cdlinm'] = self::getLineCode($item['cdlinm']);
        $item['cdtitl'] = self::getBrandCode($item['cdlinm']);
        break;
      case 'desmer':
        $item['descri'] = $item['descra'] . $item['descrb'];
        $item['cdnazi'] = $item['nazion'];
        $item['cdblco'] = 0;
        break;
      case 'dispon':
        $table = 'artcla';
        $item['tpinpu'] = 'AR';
        $item['codice'] = $item['cdartn'] . '-' . $item['cdcolo'];
        $item['tpclas'] = 'DISP';
        $item['valore'] = $item['quanti'];
        break;
      case 'gonota':
        $table = 'desart';
        $item['tpinpu'] = 'AN';
        $item['codice'] = $item['cdartn'];
        $item['tpdesc'] = $item['tpntar'];
        $item['descri'] = $item['strnot'];
        break;
      case 'gonotv':
        $table = 'desart';
        $item['tpinpu'] = 'AR';
        $item['codice'] = $item['cdarti'];
        $item['tpdesc'] = $item['tpntar'];
        $item['descri'] = $item['strnot'];
        break;
      case 'linmod':
        $descriptions = explode(' - ', $item['dslinm']);
        $item['cdserm'] = self::getSeriesCode($item['cdlinm']);
        $item['cdlinm'] = self::getLineCode($item['cdlinm']);
        $item['cdtitl'] = self::getBrandCode($item['cdlinm']);
        $item['dsserm'] = $item['cdtitl'] == 'GZ' ? (isset($descriptions[1]) ? $descriptions[1] : '') : (isset($descriptions[2]) ? $descriptions[2] : '');
        $item['dslinm'] = $item['cdtitl'] == 'GZ' ? $descriptions[0] : (isset($descriptions[1]) ? $descriptions[1] : '');
        $item['dstitl'] = $item['cdtitl'] == 'GZ' ? 'Giuseppe Zanotti' : $descriptions[0];
        break;
      case 'lsvstgl':
        $table = 'lscorp';
        $item['taglia'] = $item['strtgl'];
        break;
      case 'sequsa':
        $table = 'b2b_addinf';
        $item['tabell'] = 'tipolo';
        $item['codic1'] = $item['cdartn'];
        $item['codic2'] = '';
        $item['tpdato'] = 'seqra2';
        $item['valore'] = $item['seqrap'];
        break;
      case 'sparti':
        $item['tpcomp'] = $item['tppers'];
        break;
      case 'svcorp':
        $class = 'Svtest';
        $item['agente'] = $item['cdagen'];
        $item['catego'] = $item['cdcata'] . ' - ' . $item['dscata'];
        $item['ccitta'] = $item['ccitta'];
        $item['cdagen'] = $item['cdagen'];
        $item['cdstag'] = $item['cdstag'];
        $item['codana'] = $item['cdanag'];
        $item['codart'] = $item['cdarti'];
        $item['codcol'] = $item['cdcolo'];
        $item['codneu'] = $item['cdartn'];
        $item['datcon'] = $item['datcon'];
        $item['datord'] = $item['datord'];
        $item['desart'] = $item['dsarti'];
        $item['desneu'] = $item['dsartn'];
        $item['genere'] = $item['tpgene'] . ' - ' . $item['dstgen'];
        $item['idtest'] = $item['idtest'];
        $item['implor'] = $item['implor'];
        $item['impnet'] = $item['impnet'];
        $item['lineam'] = $item['cdlinm'] . ' - ' . $item['dslinm'];
        $item['nazion'] = $item['cdnazi'] . ' - ' . $item['dsnazi'];
        $item['numero'] = $item['idtest'];
        $item['pagame'] = $item['tppaga'] . ' - ' . $item['dspaga'];
        $item['precos'] = $item['precos'];
        $item['prelis'] = $item['prelis'];
        $item['prorig'] = $item['seqrap'];
        $item['qtatot'] = $item['quanti'];
        $item['ragsoc'] = $item['ragsoc'];
        $item['seriem'] = $item['cdserm'] . ' - ' . $item['dsserm'];
        $item['tipana'] = 'CL';
        $item['tipmod'] = $item['tpmode'] . ' - ' . $item['dstmod'];
        $item['tipord'] = $item['tpordc'];
        $item['titlin'] = $item['cdtitl'] . ' - ' . $item['dstitl'];
        $item['tporto'] = $item['tpport'] . ' - ' . $item['dstpor'];
        $item['unimis'] = $item['ummaga'];
        $item['zonage'] = $item['cdzona'] . ' - ' . $item['dszona'];
        break;
      case 'tabstg':
        $item['dsstag'] = $item['dstag'];
        break;
      case 'tipolo':
        $item['cdlinm'] = self::getLineCode($item['cdlnim']);
        $item['cdserm'] = self::getSeriesCode($item['cdlnim']);
        $item['cdtagl'] = $item['cdtagl'] != '' ? $item['cdtagl'] : 'UN';
        break;
      case 'titlin':
        $item['cdtitl'] = $item['cdttil'];
        break;
      case 'tpgene':
        $item['dsgene'] = $item['dstgen'];
        break;
      case 'tppaga':
        $item['tpindo'] = '';
        break;
    }
  }

  private static function mapTables($table) {
    $table = strtolower($table);
    switch ($table) {
      case 'budget':
        return 'b2b_budget';
      case 'gonota':
      case 'gonotv':
        return 'desart';
      case 'lsvstgl':
        return 'lscorp';
      case 'sequsa':
        return 'b2b_addinf';
      default:
        return $table;
    }
  }

  private static function getBrandCode($cdlinm) {
    return substr($cdlinm, 0, 2);
  }

  private static function getLineCode($cdlinm) {
    if(empty($cdlinm)){ return "";}
    return substr($cdlinm, 2, 3) == 'OTH' ? substr($cdlinm, 0, 5) : substr($cdlinm, 0, 3);
  }

  private static function getSeriesCode($cdlinm) {
    if(empty($cdlinm)){ return "";}
    return substr($cdlinm, 2, 3) == 'OTH' ? substr($cdlinm, 5) : substr($cdlinm, 3);
  }

  private static function preImport($echo ) {
    // B2b_budget
    $query = "CREATE TABLE IF NOT EXISTS `b2b_budget` (`tpanag` varchar(2) NOT NULL,
      `cdanag` varchar(30) NOT NULL, `cdtitl` varchar(4) NOT NULL, `cdlinm` varchar(10) NOT NULL,
      `cdserm` varchar(10) NOT NULL DEFAULT '', `cduscc` varchar(10) NOT NULL DEFAULT '',
      `cdstg0` varchar(4) NOT NULL, `qtabdg` double NOT NULL DEFAULT '0', `valbdg` double NOT NULL DEFAULT '0', `qtatrg` double NOT NULL DEFAULT '0',`valtrg` double NOT NULL DEFAULT '0',
      `cdstg1` varchar(4) NOT NULL DEFAULT '', `qtast1` double NOT NULL DEFAULT '0', `valst1` double NOT NULL DEFAULT '0',
      `cdstg2` varchar(4) NOT NULL DEFAULT '', `qtast2` double NOT NULL DEFAULT '0', `valst2` double NOT NULL DEFAULT '0',
      `cdstg3` varchar(4) NOT NULL DEFAULT '', `qtast3` double NOT NULL DEFAULT '0', `valst3` double NOT NULL DEFAULT '0',
      `cdstg4` varchar(4) NOT NULL DEFAULT '', `qtast4` double NOT NULL DEFAULT '0', `valst4` double NOT NULL DEFAULT '0',
      PRIMARY KEY (`cdanag`,`tpanag`,`cdtitl`,`cdlinm`,`cdserm`,`cduscc`,`cdstg0`)) ENGINE=InnoDB DEFAULT CHARSET=utf8";
    Di::getDefault()->get('db')->query($query);

    // Delete ctarti
    $file = self::getSyncFile('ctnoar');
    if (file_exists($file) && ($handle = fopen($file, 'r')) !== FALSE) {
      // BOM as a string for comparison.
      $bom = "\xef\xbb\xbf";

      // Progress file pointer and get first 3 characters to compare to the BOM string.
      if (fgets($handle, 4) !== $bom) {
        // BOM not found - rewind pointer to start of file.
        rewind($handle);
      }

      // Skip first row
      $data = fgetcsv($handle);

      $fp = file($file);
      $totLines = count($fp) - 1;

      $i = 1;
      while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
        // Avoid empty rows
        if ($data[0] != '') {
          $cdcata = $data[0];
          $cdarti = $data[1];
          //\Go2B\Models\Ctarti::deleteArticleFromCatalog($data[1], $data[0]); -- MODIFIED 06/09/2021
          \Go2B\Models\Ctarti::deleteArticleFromCatalog($cdcata, $cdarti);
          \Go2B\Models\B2bCtllog::insertLogOperation($cdarti, $cdcata, 2, 'DELETE', 'sync_job');

          if ($echo) {
            echo 'Delete ctarti Row ' . $i . '/' . $totLines . "\n";
            $i++;
          }
        }
      }

      fclose($handle);
      rename($file, str_replace('io/import/', 'io/import/processed/', $file));
    }

    // Log ctarti insert
    $file = self::getSyncFile('ctarti');
    if (file_exists($file) && ($handle = fopen($file, 'r')) !== FALSE) {
      // BOM as a string for comparison.
      $bom = "\xef\xbb\xbf";

      // Progress file pointer and get first 3 characters to compare to the BOM string.
      if (fgets($handle, 4) !== $bom) {
        // BOM not found - rewind pointer to start of file.
        rewind($handle);
      }

      // Skip first row
      $data = fgetcsv($handle);

      $fp = file($file);
      $totLines = count($fp) - 1;

      $i = 1;
      while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
        // Avoid empty rows
        if ($data[0] != '') {
          $cdcata = $data[0];
          $cdarti = $data[1];
          \Go2B\Models\Ctarti::deleteArticleFromCatalog($cdcata, $cdarti);
          \Go2B\Models\B2bCtllog::insertLogOperation($cdarti, $cdcata, 2, 'INSERT', 'sync_job');

          if ($echo) {
            echo 'Log ctllog insert ctarti Row ' . $i . '/' . $totLines . "\n";
            $i++;
          }
        }
      }

      fclose($handle);
    }
  }

  private static function postImport() {
    // Fill tpcomp
    echo 'Filling "tpcomp" table from "sparti" table' . "\n";
    $query = "INSERT IGNORE INTO tpcomp (tpcomp, dscomp, flstat, seqrap)
      SELECT DISTINCT tpcomp, tpcomp, 0, 0
      FROM sparti
      WHERE tpcomp != '' ";
    Di::getDefault()->get('db')->query($query);

    // Update lscorp.cdvalu
    echo 'Updating currency on lscorp ' . "\n";
    $query = 'UPDATE lscorp lc
      INNER JOIN lstest lt ON lt.nulist = lc.nulist
      SET lc.cdvalu = lt.cdvalu';
    Di::getDefault()->get('db')->query($query);

    // Check if exists custom articles
    echo 'Checking for custom article' . "\n";
    $file = self::getSyncFile('anaart');
    if (file_exists($file) && ($handle = fopen($file, 'r')) !== FALSE) {
      // Skip first row
      $data = fgetcsv($handle);
      while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
        if ($data[3] != '') {
          $newCdarti = $data[0];
          $oldCdarti = $data[3];
          $newCdcolo = $data[6];
          echo 'Substituting ' . $oldCdarti . ' with ' . $newCdarti . "\n";
          \Go2B\Models\Occorp::replaceCustomArticleWithOfficialArticle($oldCdarti, $newCdarti, $newCdcolo);
          \Go2B\Models\Anaart::blockArticle($oldCdarti);
          \Go2B\Models\Anaart::emptyFabric($newCdarti);
        }
      }
      fclose($handle);
      rename($file, str_replace('io/import/', 'io/import/processed/', $file));
    }

    // Update prices on pending orders
    echo 'Updating prices on pending orders' . "\n";
    \Go2B\Models\Octagl::updateOrderSizesPrices();

    echo 'Updating final customers on orders' . "\n";
    // Update final customers on orders
    $query = "UPDATE octest ot
      INNER JOIN anagra ag ON ag.newcod = ot.cdanag
      SET ot.tpanag = 'CL', ot.cdanag = ag.cdanag
      WHERE ot.tpanag = 'CN' AND ag.tpanag = 'CL' AND ag.cdblco = 0";
    Di::getDefault()->get('db')->query($query);

    echo 'Updating new customers with final customers' . "\n";
    // Update new customers with final customers
    $query = "UPDATE anagra ag1
      SET ag1.cdblco = 999
      WHERE ag1.tpanag = 'CN' AND ag1.cdblco = 0
      AND COALESCE(
        (SELECT COUNT(ag2.cdanag)
        FROM (SELECT * FROM anagra) AS ag2
        WHERE ag2.tpanag = 'CL' AND ag2.newcod = ag1.cdanag)
        , 0) > 0";
    Di::getDefault()->get('db')->query($query);

    echo 'Updating budgets' . "\n";
    \Go2B\Models\B2bBgtest::emptyTable();
    $query = "INSERT INTO b2b_bgtest (nubudg, tpanag, cdanag, cdtitl, cdlinm, cdserm, cduscc, cdstag, qtabdg, valbdg, qtatrg, valtrg)
      SELECT null, tpanag, cdanag, cdtitl, cdlinm, cdserm, cduscc, cdstg0, qtabdg, valbdg, qtatrg, valtrg
      FROM b2b_budget";
    Di::getDefault()->get('db')->query($query);

    \Go2B\Models\B2bBgcorp::emptyTable();
    $query = "INSERT INTO b2b_bgcorp (nubudg, cdstag, qtastg, valstg)
      SELECT bt.nubudg, bg.cdstg1, bg.qtast1, bg.valst1
      FROM b2b_budget bg
      INNER JOIN b2b_bgtest bt ON bt.tpanag = bg.tpanag AND bt.cdanag = bg.cdanag
      AND bt.cdtitl = bg.cdtitl AND bt.cdlinm = bg.cdlinm AND bt.cdserm = bg.cdserm
      AND bt.cduscc = bg.cduscc AND bt.cdstag = bg.cdstg0
      WHERE bg.cdstg1 != ''";
    Di::getDefault()->get('db')->query($query);
    $query = "INSERT INTO b2b_bgcorp (nubudg, cdstag, qtastg, valstg)
      SELECT bt.nubudg, bg.cdstg2, bg.qtast2, bg.valst2
      FROM b2b_budget bg
      INNER JOIN b2b_bgtest bt ON bt.tpanag = bg.tpanag AND bt.cdanag = bg.cdanag
      AND bt.cdtitl = bg.cdtitl AND bt.cdlinm = bg.cdlinm AND bt.cdserm = bg.cdserm
      AND bt.cduscc = bg.cduscc AND bt.cdstag = bg.cdstg0
      WHERE bg.cdstg2 != ''";
    Di::getDefault()->get('db')->query($query);
    $query = "INSERT INTO b2b_bgcorp (nubudg, cdstag, qtastg, valstg)
      SELECT bt.nubudg, bg.cdstg3, bg.qtast3, bg.valst3
      FROM b2b_budget bg
      INNER JOIN b2b_bgtest bt ON bt.tpanag = bg.tpanag AND bt.cdanag = bg.cdanag
      AND bt.cdtitl = bg.cdtitl AND bt.cdlinm = bg.cdlinm AND bt.cdserm = bg.cdserm
      AND bt.cduscc = bg.cduscc AND bt.cdstag = bg.cdstg0
      WHERE bg.cdstg3 != ''";
    Di::getDefault()->get('db')->query($query);
    $query = "INSERT INTO b2b_bgcorp (nubudg, cdstag, qtastg, valstg)
      SELECT bt.nubudg, bg.cdstg4, bg.qtast4, bg.valst4
      FROM b2b_budget bg
      INNER JOIN b2b_bgtest bt ON bt.tpanag = bg.tpanag AND bt.cdanag = bg.cdanag
      AND bt.cdtitl = bg.cdtitl AND bt.cdlinm = bg.cdlinm AND bt.cdserm = bg.cdserm
      AND bt.cduscc = bg.cduscc AND bt.cdstag = bg.cdstg0
      WHERE bg.cdstg4 != ''";
    Di::getDefault()->get('db')->query($query);
    $query = "DROP TABLE b2b_budget";
    Di::getDefault()->get('db')->query($query);
  }
}

?>
