<?php

namespace Go2B\Plugins;

use Exception;
use Go2B\Models\Octest;
use Phalcon\Logger\Adapter\File as FileAdapter;
use SoapClient;
use SoapFault;
use stdClass;

/**
 * Api Manager per NAV
 */
class ApiManager
{
    const AUTH_NTLM = "NTLM";
    const AUTH_AUTH = "BASIC";
    const METHOD_POST = "POST";
    const METHOD_GET = "GET";
    const METHOD_UPDATE = "UPDATE";
    const METHOD_DELETE = "DELETE";
    const METHOD_PATCH = "PATCH";
    const STATUS_ORDER_SOAP_NOT_FOUND = "ORDER NOT FOUND";
    const STATUS_ORDER_SOAP_SUCCESS = "SUCCESS";

    protected $di;

    protected $navConfig;

    protected $logger;

    public function __construct()
    {
        $this->di = \Phalcon\Di\FactoryDefault::getDefault();
        $config = $this->di->get('config');
        $this->navConfig = $config->get('nav');

        $targetPath = APP_PATH . $config->application->logDir;

        if (!empty($targetPath)) {
            // Trova tutti i file di log presenti nella directory
            $logFiles = glob($targetPath . 'nav-*.log');

            // Ordina i file per data di modifica
            usort($logFiles, function ($a, $b) {
                return filemtime($a) < filemtime($b);
            });

            // Rimuovi i file di log che superano il limite massimo
            while (count($logFiles) >= 12) {
                $fileToRemove = array_pop($logFiles);
                unlink($fileToRemove);
            }

            if ($this->navConfig->logEnabled) {
                // Init Custom Logger
                if (!is_dir($targetPath) && !mkdir($targetPath, 0777, true)) {
                    $baseLogger = $this->di->get('logger');
                    if ($baseLogger) {
                        $baseLogger->error("Error creating folder $targetPath for nav custom log");
                        $this->logger = $baseLogger;
                    }
                } else {
                    $logFilename = $targetPath . 'nav-' . date('Y-m-d') . '.log';
                    if (!file_exists($logFilename)) {
                        // Crea un nuovo file di log per il giorno corrente
                        file_put_contents($logFilename, '', FILE_APPEND);
                        chmod($logFilename, 0777);
                    }

                    $this->logger = new FileAdapter($logFilename);
                }
            }
        }
    }

    /**
     * Verify the required parameter 'company_id' is set
     *
     * @param string $company_id id for company
     * @throws \InvalidArgumentException
     */
    public static function checkCompanyId($company_id)
    {
        if ($company_id === null || (is_array($company_id) && count($company_id) === 0)) {
            throw new \InvalidArgumentException(
                'Missing the required parameter $company_id when calling listCustomers'
            );
        }
    }

    /**
     * Create string of parameters for GET queries
     *
     * @param string $company_id id for company
     * @return string
     */
    public static function getQueryParamsString($queryParams)
    {
        $queryParamsString = '';
        if (!empty($queryParams)) {
            foreach ($queryParams as $key => $value) {
                $value = is_array($value) ? self::serializeCollection($value, 'csv', true) : $value;
                if (!empty($value)) {
                    $queryParamsString .= ($queryParamsString != '' ? '&' : '?') . '$' . $key . '=' . self::toQueryValue($value);
                }
            }
        }
        return $queryParamsString;
    }

    /**
     * Serialize an array to a string.
     *
     * @param array $collection collection to serialize to a string
     * @param string $collectionFormat the format use for serialization (csv,
     * ssv, tsv, pipes, multi)
     * @param bool $allowCollectionFormatMulti allow collection format to be a multidimensional array
     *
     * @return string
     */
    public static function serializeCollection(array $collection, $collectionFormat, $allowCollectionFormatMulti = false)
    {
        if ($allowCollectionFormatMulti && ('multi' === $collectionFormat)) {
            // http_build_query() almost does the job for us. We just
            // need to fix the result of multidimensional arrays.
            return preg_replace('/%5B[0-9]+%5D=/', '=', http_build_query($collection, '', '&'));
        }
        switch ($collectionFormat) {
            case 'pipes':
                return implode('|', $collection);
            case 'tsv':
                return implode("\t", $collection);
            case 'ssv':
                return implode(' ', $collection);
            case 'csv':
                // Deliberate fall through. CSV is default format.
            default:
                return implode(',', $collection);
        }
    }

    /**
     * Take value and turn it into a string suitable for inclusion in
     * the query, by imploding comma-separated if it's an object.
     * If it's a string, pass through unchanged. It will be url-encoded
     * later.
     *
     * @param string[]|string|\DateTime $object an object to be serialized to a string
     *
     * @return string the serialized object
     */
    public static function toQueryValue($object)
    {
        if (is_array($object)) {
            return implode(',', $object);
        } else {
            return self::toString($object);
        }
    }

    /**
     * Take value and turn it into a string suitable for inclusion in
     * the parameter. If it's a string, pass through unchanged
     * If it's a datetime object, format it in ISO8601
     *
     * @param string|\DateTime $value the value of the parameter
     *
     * @return string the header string
     */
    public static function toString($value)
    {
        if ($value instanceof \DateTime) { // datetime in ISO8601 format
            return urlencode($value->format(\DateTime::ATOM));
        } else {
            return urlencode($value);
        }
    }

    /**
     * Execute a curl to Nav with a request
     */
    public function requestToNav($method, $api, $payload = null, $etag = null, $useAlternativePath = false)
    {
        try {
            while (substr($api, 0, 1) == "/") {
                $api = substr($api, 1);
            }

            $domain = $this->navConfig->domain;
            $auth = $this->navConfig->auth;
            $url = $this->navConfig->baseUrl;
            $username = $this->navConfig->username;
            $password = $this->navConfig->password;

            if ($useAlternativePath) {
                if (!empty($this->navConfig->dynamicUrl)) {
                    $url = $this->navConfig->dynamicUrl;
                } else {
                    $url = str_replace('NBL', 'DYN', $url);
                }
            }

            $endPoint = $url . $api;

            $curl = curl_init($endPoint);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
            curl_setopt($curl, CURLOPT_ENCODING, '');
            curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);

            $httpHeader = $etag == null
                ? ['Content-Type: application/json']
                : ['Content-Type: application/json', 'If-Match:' . $etag];

            if ($this->logger) {
                $this->logger->debug('richiesta su ' . $endPoint);
            }

            switch ($auth) {
                case self::AUTH_NTLM:
                    // Set NTLLM authentication
                    define('USERPWD', (!empty($domain) ? $domain . "\\" : null) . $username . ':' . $password);
                    curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_NTLM);
                    curl_setopt($curl, CURLOPT_USERPWD, USERPWD);
                    if ($this->logger) {
                        $this->logger->debug('Auth: ' . USERPWD);
                    }
                    break;
                case self::AUTH_AUTH:
                default:
                    $httpHeader[] = "Authorization: Basic " . base64_encode($username . ":" . $password);
                    if ($this->logger) {
                        $this->logger->debug('Auth: ' .  "Authorization: Basic " . base64_encode($username . ":" . $password));
                    }
                    break;
            }

            curl_setopt($curl, CURLOPT_HTTPHEADER, $httpHeader);

            // Set correct request method
            if (in_array($method, ['GET', 'PATCH', 'POST', 'DELETE'])) {
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);
            }

            // Set payload (e.g. "POST" method) if needed
            if ($payload != null) {
                if ($this->logger) {
                    $this->logger->info("PAYLOAD: " . ($payload instanceof stdClass ? json_encode($payload) : $payload));
                }

                curl_setopt($curl, CURLOPT_POSTFIELDS, ($payload instanceof stdClass ? json_encode($payload) : $payload));
            }

            // Execute and close
            $response = curl_exec($curl);
            $contentType = curl_getinfo($curl, CURLINFO_CONTENT_TYPE);

            curl_close($curl);

            if (strpos($contentType, "application/json") !== false) {
                $response = $response != null ? json_decode($response) : null;
                return isset($response->value) ? $response->value : $response;
            }

            return $response;
        } catch (\Exception $ex) {
            if ($this->logger) {
                $this->logger->error(print_r($ex->getMessage(), true));
            }
        }

        return null;
    }

    /**
     * Invia l'ordine a BC tramite SOAP
     *
     * @param string $url
     * @param array $credendials
     * @param string $callMethod
     * @param array $params
     * @return stdClass|null
     * @throws SoapFault
     */

    protected function sendToSoap($url, $callMethod, $credendials = null, $params = null)
    {
        $options = null;
        if (count($credendials) >= 1) {
            $options = $credendials;
        }

        /** @var SoapClient $soap */
        $soap = new SoapClient(
            $url,
            $options
        );

        $paramsXmlSoap = array();

        if (!empty($params)) {
            $paramsXmlSoap = $params;
        }

        $response = $soap->__soapCall(
            $callMethod,
            array(
                $paramsXmlSoap
            )
        );

        if ($response instanceof stdClass) {
            return $response;
        } else if ($this->logger) {
            $this->logger->error("ALTRA RISPOSTA" . print_r($response, true));
        }

        return null;
    }

    protected function sendOrderToSoap($orderID)
    {
        $xmlWrite = '<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/" xmlns:nbl="urn:microsoft-dynamics-schemas/codeunit/NBLImportOrder">
<soapenv:Header/>
<soapenv:Body>
    <nbl:ImportOrder>
        <nbl:orderID>' . $orderID . '</nbl:orderID>
    </nbl:ImportOrder>
</soapenv:Body>
</soapenv:Envelope>';

        $curl = curl_init();
        $url = $this->navConfig->orderUrl;
        $username = $this->navConfig->username;
        $password = $this->navConfig->password;

        $this->logger->debug("sendOrderToSoap: " . print_r($url, true));

        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => $xmlWrite,
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/xml',
                'SOAPAction: "#POST"',
                'Authorization: Basic ' . base64_encode($username . ":" . $password)
            ),
        ));

        $response = curl_exec($curl);
        curl_close($curl);

        $this->logger->debug("RISPOSTA NAV " . print_r($response, true));
        return true;
    }
}
